#!BPY 

""" 
Name: 'ASCII Scene (.ase) v0.6.2' 
Blender: 240 
Group: 'Export' 
Tooltip: 'ASCII Scene Export (*.ase)' 
""" 
__author__ = "Goofos" 
__version__ = "0.6.2" 
__bpydoc__ = """\ 
-- ASCII Scene (.ase) export script v0.6.2 --<br> 
Can export:<br> 
-Mesh Objects<br> 
-Materials and Textures (no Procedural but Image)<br> 
    Note: Normalmaps will be exported as Bumpmaps<br> 
    Currently supported: Amb, Col, Csp, Hard, Alpha, Nor, Disp<br> 
-Vertex Colors<br> 
    Note: If the mesh has materials you must toggle<br> 
          "Vcol Paint" on in Material tab. Without<br> 
          Materials, make sure "VertCol" in Mesh tab<br> 
          is on. 
-Face UV<br> 
    Make sure "TexFace" in Mesh tab is on.<br> 
-Some other things :><br> 
<br> 
-- Export Options Description --<br> 
Apply Modifiers: Export the mesh with applied modifiers.<br> 
Materials: Export Materials if any.<br> 
Face UV: Export TexFace UV if any.<br> 
Vertex Colors: Export Vertex Colors if any (See note above).<br> 
Selection Only: Export only selected Objects or all Objects.<br> 
UV Image as Material: Export the images assigned in UV editor<br> 
    as Materials. Only works if the Object has no Materials.<br> 
VertGr. as SmoothGr.: You can export SmoothGroups defined by<br> 
    VertexGroups. Simply create a VertGroup and name it<br> 
    "smooth." plus a group number, f.e. "smooth.2"<br> 
    Note: Will not work if "Apply Modifiers" is used.<br> 
""" 
# goofos at epruegel.de 
# 
# ***** BEGIN GPL LICENSE BLOCK ***** 
# 
# This program is free software; you can redistribute it and/or 
# modify it under the terms of the GNU General Public License 
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version. 
# 
# This program is distributed in the hope that it will be useful, 
# but WITHOUT ANY WARRANTY; without even the implied warranty of 
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License 
# along with this program; if not, write to the Free Software Foundation, 
# Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. 
# 
# ***** END GPL LICENCE BLOCK ***** 

import Blender, time, os, sys as osSys 
from Blender import sys, Window, Draw, Scene, Mesh, Material, Texture, Image, Mathutils 
#import meshtools 


#============================================ 
#           Write! 
#============================================ 

def write(filename): 
   start = time.clock() 
   print_boxed('---------Start of Export------------') 
   print 'Export Path: ' + filename 

   global exp_list, Tab, idnt, imgTable, worldTable 

   exp_list =[] 
   Tab = "\t" 
   idnt = 1 
   imgTable = {} 
   worldTable = {'ambR': 0.0, 'ambG': 0.0, 'ambB': 0.0, 'horR': 0.0, 'horG': 0.0, 'horB': 0.0} #default 
   total = {'Verts': 0, 'Tris': 0, 'Faces': 0} 

   file = open(filename, "w") 
   set_up(file, exp_list, imgTable, worldTable) 
   write_header(file, filename, worldTable) 
   write_materials(file, exp_list, worldTable) 
   write_mesh(file, exp_list, imgTable, total) 
   file.close() 
    
   Blender.Window.DrawProgressBar(0, "")    # clear progressbar 
   end = time.clock() 
   seconds = " in %.2f %s" % (end-start, "seconds") 
   totals = "Verts: %i Tris: %i Faces: %i" % (total['Verts'], total['Tris'], total['Faces']) 
   print_boxed(totals) 
   message = "Successfully exported " + os.path.basename(filename) + seconds 
   #meshtools.print_boxed(message) 
   print_boxed(message) 


def print_boxed(text): #Copy/Paste from meshtools, only to remove the beep :) 
   lines = text.splitlines() 
   maxlinelen = max(map(len, lines)) 
   if osSys.platform[:3] == "win": 
      print chr(218)+chr(196) + chr(196)*maxlinelen + chr(196)+chr(191) 
      for line in lines: 
         print chr(179) + ' ' + line.ljust(maxlinelen) + ' ' + chr(179) 
      print chr(192)+chr(196) + chr(196)*maxlinelen + chr(196)+chr(217) 
   else: 
      print '+-' + '-'*maxlinelen + '-+' 
      for line in lines: print '| ' + line.ljust(maxlinelen) + ' |' 
      print '+-' + '-'*maxlinelen + '-+' 
   #print '\a\r', # beep when done 

#============================================ 
#           Setup 
#============================================ 

def set_up(file, exp_list, imgTable, worldTable): 
   print "Setup" 

   #Get selected Objects or if none selected all Objects from the current Scene 
   if Blender.Object.GetSelected() and guiTable['SELO'] == 1: 
      objects = Blender.Object.GetSelected() 
   else: 
      if Blender.Object.Get(): 
         objects = Blender.Object.Get() 

   set_lists(file, exp_list, objects, imgTable, worldTable) 


def set_lists(file, exp_list, objects, imgTable, worldTable): 
   global mat_cnt 
   mat_cnt = 0 
   mat_ref = 0 

   #exp_list = [container1 = [ [mesh], [materials] or [uvimages], [material_ref], [mat_type] ],...] 

   for current_obj in objects: 
      container = [] 
      if current_obj.getType() == 'Mesh': 
         container.append(current_obj) 

         mat_type = 0 #1=Material, 2=UV Images 
         mesh = current_obj.data 

         #Find used Materials by the Meshes 
         if guiTable['MTL'] == 1 and mesh.materials or current_obj.getMaterials(0): #Materials 
            mat_type = 1 
            mesh_mats = [] 
            me_mats = mesh.materials 
            ob_mats = current_obj.getMaterials(0) 

            if (mesh.materials): #ME assigned Mats 
               for current_mat in me_mats: 
                  mesh_mats.append(current_mat) 

            if (current_obj.getMaterials(0)): #OB assigned Mats 
               for current_mat in ob_mats: 
                  mesh_mats.append(current_mat) 

            container.append(mesh_mats) 
            container.append(mat_ref) 
            container.append(mat_type) 
            mat_cnt+=1 
            mat_ref+=1 

         else: #UV-Editor assigned Images 
            if guiTable['UVI'] == 1: 
               face_img = [] 
               img_idx = 0 

               for current_face in mesh.faces: 
                  if current_face.image: 
                     if current_face.image not in face_img: 
                        face_img.append(current_face.image) 
                        cur_img = current_face.image.getName() 
                        imgTable[cur_img] = img_idx 
                        img_idx += 1 

               if len(face_img) is not 0: 
                  mat_type = 2 
                  container.append(face_img) 
                  container.append(mat_ref) 
                  container.append(mat_type) 
                  mat_ref+=1 
                  mat_cnt+=1 

         exp_list.append(container) 

   #If there is a world shader get some values 
   world = Blender.World.GetCurrent() 
   if world != None: 
      worldAmb = world.getAmb() 
      worldHor = world.getHor() 

      worldTable['ambR'] = worldAmb[0] 
      worldTable['ambG'] = worldAmb[1] 
      worldTable['ambB'] = worldAmb[2] 

      worldTable['horR'] = worldHor[0] 
      worldTable['horG'] = worldHor[1] 
      worldTable['horB'] = worldHor[2] 



#============================================ 
#           Header/Scene 
#============================================ 

def write_header(file, filename, worldTable): 
   print "Write Header" 

   scn = Scene.GetCurrent() 
   context = scn.getRenderingContext() 

   file.write("*3DSMAX_ASCIIEXPORT%s200\n" % (Tab)) 
   file.write("*COMMENT \"Exported from Blender %s - %s\"\n" % (Blender.Get('version'), time.asctime(time.localtime()))) 
   file.write("*SCENE {\n") 
   file.write("%s*SCENE_FILENAME \"%s\"\n" % (Tab, os.path.basename(Blender.Get('filename')))) 

   file.write("%s*SCENE_FIRSTFRAME %d\n" % (Tab,context.startFrame())) 
   file.write("%s*SCENE_LASTFRAME %d\n" % (Tab,context.endFrame())) 
   file.write("%s*SCENE_FRAMESPEED %d\n" % (Tab,context.framesPerSec())) 
   file.write("%s*SCENE_TICKSPERFRAME 160\n" % (Tab)) #Blender have no Ticks? 

   file.write("%s*SCENE_BACKGROUND_STATIC %.4f %.4f %.4f\n" % (Tab, worldTable['horR'], worldTable['horG'], worldTable['horB'])) 
   file.write("%s*SCENE_AMBIENT_STATIC %.4f %.4f %.4f\n" % (Tab, worldTable['ambR'], worldTable['ambG'], worldTable['ambB'])) 
   file.write("}\n") 


#============================================ 
#           Materials 
#============================================ 

def write_materials(file, exp_list, worldTable): 
   print "Write Materials" 

   file.write("*MATERIAL_LIST {\n") 
   file.write("%s*MATERIAL_COUNT %s\n" % (Tab, mat_cnt)) 

   for current_container in exp_list: 
       
      if len(current_container) == 4: # Materials exist 

         if len(current_container[1]) == 1: # single mat 
            mat_class = 'Standard' 

            if current_container[3] == 1: # Blender material 
               mats = current_container[1] 
               material = mats[0] 
               mat_name = material.name 

               file.write("%s*MATERIAL %d {\n" % ((Tab), current_container[2])) 

               idnt = 2 
               mat_para(file, idnt, material, mat_name, mat_class, worldTable) 
               mat_dummy(file, idnt) 
               mat_map(file, idnt, mat_name) 

               file.write("%s}\n" % (Tab)) 

            elif current_container[3] == 2: # UV fake mat 
               uv_imgs = current_container[1] 
               uv_image = uv_imgs[0] 
               uv_name = uv_image.getName() 

               file.write("%s*MATERIAL %d {\n" % ((Tab), current_container[2])) 

               idnt = 2 
               mat_uv(file, idnt, uv_image, uv_name, mat_class, worldTable) 
               mat_dummy(file, idnt) 
               map_uv(file, idnt, uv_image, uv_name) 

               file.write("%s}\n" % (Tab)) 

         elif len(current_container[1]) > 1: # multiple mat 
            mat_class = 'Multi/Sub-Object' 

            if current_container[3] == 1: # Blender materials 
               mats = current_container[1] 
               material = mats[0] 
               mat_name = 'Multi # ' + material.name 
               submat_no = len(mats) 
               submat_cnt = 0 

               idnt = 2 
               file.write("%s*MATERIAL %d {\n" % ((Tab), current_container[2])) 

               mat_para(file, idnt, material, mat_name, mat_class, worldTable) 

               file.write("%s*NUMSUBMTLS %d\n" % ((Tab*idnt), submat_no)) 

               for current_mat in mats: 
                  material = current_mat 
                  mat_class = 'Standard' 
                  mat_name = material.name 

                  idnt = 2 
                  file.write("%s*SUBMATERIAL %d {\n" % ((Tab*idnt), submat_cnt)) 
                  submat_cnt += 1 

                  idnt = 3 
                  mat_para(file, idnt, material, mat_name, mat_class, worldTable) 
                  mat_dummy(file, idnt) 
                  mat_map(file, idnt, mat_name) 

                  idnt = 2 
                  file.write("%s}\n" % (Tab*idnt)) 

               file.write("%s}\n" % (Tab)) 


            elif current_container[3] == 2: # UV fake mats 
               uv_imgs = current_container[1] 
               uv_image = uv_imgs[0] 
               uv_name = 'Multi \# ' + uv_image.getName() 
               submat_no = len(uv_imgs) 
               submat_cnt = 0 

               idnt = 2 
               file.write("%s*MATERIAL %d {\n" % ((Tab), current_container[2])) 

               mat_uv(file, idnt, uv_image, uv_name, mat_class, worldTable) 

               file.write("%s*NUMSUBMTLS %d\n" % ((Tab*idnt), submat_no)) 
               for current_uv in uv_imgs: 
                  uv_image = current_uv 
                  mat_class = 'Standard' 
                  uv_name = uv_image.getName() 

                  idnt = 2 
                  file.write("%s*SUBMATERIAL %d {\n" % ((Tab*idnt), submat_cnt)) 
                  submat_cnt += 1 

                  idnt = 3 
                  mat_uv(file, idnt, uv_image, uv_name, mat_class, worldTable) 
                  mat_dummy(file, idnt) 
                  map_uv(file, idnt, uv_image, uv_name) 

                  idnt = 2 
                  file.write("%s}\n" % (Tab*idnt)) 

               file.write("%s}\n" % (Tab)) 


   file.write("}\n") 


def mat_para(file, idnt, material, mat_name, mat_class, worldTable): 

   mat_amb = material.getAmb() 
   mat_dif = material.getRGBCol() 
   mat_specCol = material.getSpecCol() 
   mat_spec = material.getSpec() 
   mat_hard = material.getHardness() 
   mat_alpha = 1.0000-material.getAlpha() 

   file.write("%s*MATERIAL_NAME \"%s\"\n" % ((Tab*idnt), mat_name)) 
   file.write("%s*MATERIAL_CLASS \"%s\"\n" % ((Tab*idnt), mat_class)) 
   file.write("%s*MATERIAL_AMBIENT %.4f   %.4f   %.4f\n" % ((Tab*idnt), (worldTable['ambR']*mat_amb), (worldTable['ambG']*mat_amb), (worldTable['ambB']*mat_amb))) #-Usefull? 
   file.write("%s*MATERIAL_DIFFUSE %.4f   %.4f   %.4f\n" % ((Tab*idnt), mat_dif[0], mat_dif[1], mat_dif[2])) 
   file.write("%s*MATERIAL_SPECULAR %.4f   %.4f   %.4f\n" % ((Tab*idnt), mat_specCol[0], mat_specCol[1], mat_specCol[2])) 
   file.write("%s*MATERIAL_SHINE %.4f\n" % ((Tab*idnt), mat_spec)) 
   file.write("%s*MATERIAL_SHINESTRENGTH %.4f\n" % ((Tab*idnt), (mat_hard/511.))) #-511 or 512? 
   file.write("%s*MATERIAL_TRANSPARENCY %.4f\n" % ((Tab*idnt), mat_alpha)) 
   file.write("%s*MATERIAL_WIRESIZE 1.0000\n" % (Tab*idnt)) 


def mat_dummy(file, idnt): 

   file.write("%s*MATERIAL_SHADING Blinn\n" % (Tab*idnt)) 
   file.write("%s*MATERIAL_XP_FALLOFF 0.0000\n" % (Tab*idnt)) 
   file.write("%s*MATERIAL_SELFILLUM 0.0000\n" % (Tab*idnt)) 
   file.write("%s*MATERIAL_FALLOFF In\n" % (Tab*idnt)) 
   file.write("%s*MATERIAL_XP_TYPE Filter\n" % (Tab*idnt)) 


def mat_map(file, idnt, mat_name): 

   mapTable = {0:'*MAP_AMBIENT',1:'*MAP_DIFFUSE',2:'*MAP_SPECULAR',3:'*MAP_SHINE',4:'*MAP_SHINESTRENGTH',5:'*MAP_SELFILLUM',6:'*MAP_OPACITY',7:'*MAP_FILTERCOLOR',8:'*MAP_BUMP',9:'*MAP_REFLECT',10:'*MAP_REFRACT',11:'*MAP_REFRACT'} 
   tex_list = [[],[],[],[],[],[],[],[],[],[],[],[]] 

   mat = Material.Get(mat_name) 
   MTexes = mat.getTextures() 

   for current_MTex in MTexes: 
      if current_MTex is not None: 
         # MAP_SUBNO 0 = *MAP_AMBIENT 
         if current_MTex.mapto & Texture.MapTo.AMB: 
            map_getTex(current_MTex, 0, (current_MTex.dvar*current_MTex.varfac), tex_list) 
         # MAP_SUBNO 1 = *MAP_DIFFUSE = COL = 1 
         elif current_MTex.mapto & Texture.MapTo.COL: 
            map_getTex(current_MTex, 1, current_MTex.colfac, tex_list) 
         # MAP_SUBNO 2 = *MAP_SPECULAR (Color)= CSP or SPEC? = 4 
         elif current_MTex.mapto & Texture.MapTo.CSP: 
            map_getTex(current_MTex, 2, current_MTex.colfac, tex_list) 
         # MAP_SUBNO 3 = *MAP_SHINE (Spec Level) = SPEC or CSP? = 32 
         elif current_MTex.mapto & Texture.MapTo.SPEC: 
            map_getTex(current_MTex, 3, (current_MTex.dvar*current_MTex.varfac), tex_list) 
         # MAP_SUBNO 4 = *MAP_SHINESTRENGTH (Gloss) = HARD = 256 
         elif current_MTex.mapto & Texture.MapTo.HARD: 
            map_getTex(current_MTex, 4, (current_MTex.dvar*current_MTex.varfac), tex_list) 
         # MAP_SUBNO 5 = *MAP_SELFILLUM 
         # MAP_SUBNO 6 = *MAP_OPACITY = ALPHA = 128 
         elif current_MTex.mapto & Texture.MapTo.ALPHA: 
            map_getTex(current_MTex, 6, (current_MTex.dvar*current_MTex.varfac), tex_list) 
         # MAP_SUBNO 7 = *MAP_FILTERCOLOR 
         # MAP_SUBNO 8 = *MAP_BUMP = NOR = 2 
         elif current_MTex.mapto & Texture.MapTo.NOR: 
            map_getTex(current_MTex, 8, (current_MTex.norfac/25), tex_list) 
         # MAP_SUBNO 9 = *MAP_REFLECT 
         elif current_MTex.mapto & Texture.MapTo.REF: 
            map_getTex(current_MTex, 9, (current_MTex.norfac/25), tex_list) 
         # MAP_SUBNO 10 = *MAP_REFRACT (refraction) 
         # MAP_SUBNO 11 = *MAP_REFRACT (displacement) 
         elif current_MTex.mapto & Texture.MapTo.DISP: 
            map_getTex(current_MTex, 11, (current_MTex.norfac/25), tex_list) 

   # Write maps 
   for current_LI in tex_list: 
      subNo = tex_list.index(current_LI) 
      for current_MTex in current_LI: 
         tex = current_MTex[0].tex 
         if tex.type == Texture.Types.IMAGE: 
            map_image(file, idnt, current_MTex, subNo, tex, mapTable[subNo]) 


def map_getTex(MTex, map_subNo, map_amount, texes): 
   # container = [[[MTex], [map_amount]], ...] 
   container = [] 
   container.append(MTex) 
   container.append(map_amount) 
   texes[map_subNo].append(container) 

          
def map_image(file, idnt, MTexCon, subNo, tex, mapType): 

   img = tex.getImage() 
   path = sys.expandpath(img.getFilename()).replace('/', '\\') 
   tex_class = 'Bitmap' 
   tex_mapType = 'Screen' 
   tex_filter = 'Pyramidal' 

   file.write("%s%s {\n" % ((Tab*idnt), mapType)) 

   idnt += 1 
   file.write("%s*MAP_NAME \"%s\"\n" % ((Tab*idnt), tex.getName())) 
   file.write("%s*MAP_CLASS \"%s\"\n" % ((Tab*idnt), tex_class)) 
   file.write("%s*MAP_SUBNO %s\n" % ((Tab*idnt), subNo)) 
   file.write("%s*MAP_AMOUNT %.4f\n" % ((Tab*idnt), MTexCon[1])) 
   file.write("%s*BITMAP \"%s\"\n" % ((Tab*idnt), path)) 
   file.write("%s*MAP_TYPE %s\n" % ((Tab*idnt), tex_mapType)) 

   # hope this part is right! 
   u_tiling = tex.repeat[0]*tex.crop[2] 
   v_tiling = tex.repeat[1]*tex.crop[3] 
   file.write("%s*UVW_U_OFFSET %.4f\n" % ((Tab*idnt), tex.crop[0])) 
   file.write("%s*UVW_V_OFFSET %.4f\n" % ((Tab*idnt), tex.crop[1])) 
   file.write("%s*UVW_U_TILING %.4f\n" % ((Tab*idnt), u_tiling)) 
   file.write("%s*UVW_V_TILING %.4f\n" % ((Tab*idnt), v_tiling)) 

   map_uvw(file, idnt) #hardcoded 

   file.write("%s*BITMAP_FILTER %s\n" % ((Tab*idnt), tex_filter)) 

   idnt -= 1 
   file.write("%s}\n" % (Tab*idnt)) 


def mat_uv(file, idnt, uv_image, uv_name, mat_class, worldTable): 
   fake_val0 = '0.0000' 
   fake_val1 = '0.1000' 
   fake_val2 = '0.5882' 
   fake_val3 = '0.9000' 
   fake_val4 = '1.0000' 

   file.write("%s*MATERIAL_NAME \"%s\"\n" % ((Tab*idnt), uv_name)) 
   file.write("%s*MATERIAL_CLASS \"%s\"\n" % ((Tab*idnt), mat_class)) 
   file.write("%s*MATERIAL_AMBIENT %.4f   %.4f   %.4f\n" % ((Tab*idnt), worldTable['ambR'], worldTable['ambG'], worldTable['ambB'])) #------------Usefull? 
   file.write("%s*MATERIAL_DIFFUSE %s   %s   %s\n" % ((Tab*idnt), fake_val2, fake_val2, fake_val2)) 
   file.write("%s*MATERIAL_SPECULAR %s   %s   %s\n" % ((Tab*idnt), fake_val3, fake_val3, fake_val3)) 
   file.write("%s*MATERIAL_SHINE %s\n" % ((Tab*idnt), fake_val1)) 
   file.write("%s*MATERIAL_SHINESTRENGTH %s\n" % ((Tab*idnt), fake_val0)) 
   file.write("%s*MATERIAL_TRANSPARENCY %s\n" % ((Tab*idnt), fake_val0)) 
   file.write("%s*MATERIAL_WIRESIZE %s\n" % ((Tab*idnt), fake_val4)) 


def map_uv(file, idnt, uv_image, uv_name): 
   map_type = '*MAP_DIFFUSE' 
   map_subNo = '1' 
   tex_class = 'Bitmap' 
   tex_mapType = 'Screen' 
   tex_filter = 'Pyramidal' 

   fake_val0 = '0.0000' 
   fake_val1 = '0.1000' 
   fake_val2 = '0.5882' 
   fake_val3 = '0.9000' 
   fake_val4 = '1.0000' 

   #replace "/" with "\" in image path 
   uv_filename = uv_image.getFilename().replace('/', '\\') 

   file.write("%s%s {\n" % ((Tab*idnt), map_type)) 

   idnt += 1 
   file.write("%s*MAP_NAME \"%s\"\n" % ((Tab*idnt), uv_name)) 
   file.write("%s*MAP_CLASS \"%s\"\n" % ((Tab*idnt), tex_class)) 
   file.write("%s*MAP_SUBNO %s\n" % ((Tab*idnt), map_subNo)) 
   file.write("%s*MAP_AMOUNT %s\n" % ((Tab*idnt), fake_val4)) 
   file.write("%s*BITMAP \"%s\"\n" % ((Tab*idnt), uv_filename)) 
   file.write("%s*MAP_TYPE %s\n" % ((Tab*idnt), tex_mapType)) 
   file.write("%s*UVW_U_OFFSET %s\n" % ((Tab*idnt), fake_val0)) 
   file.write("%s*UVW_V_OFFSET %s\n" % ((Tab*idnt), fake_val0)) 
   file.write("%s*UVW_U_TILING %s\n" % ((Tab*idnt), fake_val4)) 
   file.write("%s*UVW_V_TILING %s\n" % ((Tab*idnt), fake_val4)) 

   map_uvw(file, idnt) #hardcoded 

   file.write("%s*BITMAP_FILTER %s\n" % ((Tab*idnt), tex_filter)) 

   idnt -= 1 
   file.write("%s}\n" % (Tab*idnt)) 


def map_uvw(file, idnt): 

   fake_val0 = '0.0000' 
   fake_val1 = '1.0000' 

   file.write("%s*UVW_ANGLE %s\n" % ((Tab*idnt), fake_val0)) 
   file.write("%s*UVW_BLUR %s\n" % ((Tab*idnt), fake_val1)) 
   file.write("%s*UVW_BLUR_OFFSET %s\n" % ((Tab*idnt), fake_val0)) 
   file.write("%s*UVW_NOUSE_AMT %s\n" % ((Tab*idnt), fake_val1)) 
   file.write("%s*UVW_NOISE_SIZE %s\n" % ((Tab*idnt), fake_val1)) 
   file.write("%s*UVW_NOISE_LEVEL 1\n" % (Tab*idnt)) 
   file.write("%s*UVW_NOISE_PHASE %s\n" % ((Tab*idnt), fake_val0)) 


#============================================ 
#           Mesh 
#============================================ 


def write_mesh(file, exp_list, imgTable, total): 
   print "Write Geometric" 

   for current_container in exp_list: 

      TransTable = {'SizeX': 1, 'SizeY': 1, 'SizeZ': 1} 
      nameMe = {'objName': 'obj', 'meName': 'me'} 
      sGroups = {} 
      hasTable = {'hasMat': 0, 'hasSG': 0, 'hasUV': 0, 'hasVC': 0, 'matRef': 0} 
      count = {'face': 0, 'vert': 0, 'cVert': 0} 

      obj = current_container[0] 

      nameMe['objName'] = obj.name 
      nameMe['meName'] = obj.getData(0,1).name 

      if obj.getParent(): 
         nameMe['parent'] = obj.getParent().name 
       
      me = Mesh.New()      # Create a new mesh 

      if guiTable['MOD'] == 1:   # Use modified mesh 
         me.getFromObject(obj.name, 0) # Get the object's mesh data, cage 0 = apply mod 
      else: 
         me.getFromObject(obj.name, 1) 

      me.transform(obj.matrix)   #ASE stores transformed mesh data 
      tempObj = Blender.Object.New('Mesh', 'ASE_export_temp_obj') 
      tempObj.setMatrix(obj.matrix) 
      tempObj.link(me) 
      obj = tempObj 

      faces = me.faces 
      verts = me.verts 
       
      #Blender.Redraw() 
      count['vert'] = len(verts) 
      total['Verts'] += count['vert'] 

      if count['vert'] == 0: 
         print 'Error: ' + nameMe['meName'] + 'has 0 Verts' 
         continue 

      if len(current_container) > 1: 
         hasTable['hasMat'] = current_container[3] 
         hasTable['matRef'] = current_container[2] 

      vGroups = me.getVertGroupNames() 
      if guiTable['VG2SG'] == 1 and len(vGroups) > 0: 
         for current_VG in vGroups: 
            if current_VG.lower().count("smooth."): 
               hasTable['hasSG'] = 1 
               gWeight = current_VG.replace("smooth.", "") 
               sGroups[current_VG] = gWeight 

      if guiTable['UV'] == 1: 
         if me.faceUV == True or me.faceUV == 1: 
            hasTable['hasUV'] = 1 

      if guiTable['VC'] == 1: 
         if me.vertexColors == True or me.vertexColors == 1: 
            hasTable['hasVC'] = 1 
         elif hasTable['hasMat'] == 1: # Blender material 
            mats = current_container[1] 
            for current_mat in mats: 
               if current_mat.getMode() & Material.Modes['VCOL_PAINT']: 
                  hasTable['hasVC'] = 1 
                  break 

      for current_face in faces: 
         if len(current_face.verts) is 3: 
            count['face'] += 1 
            total['Tris'] += 1 
            total['Faces'] += 1 
         elif len(current_face.verts) is 4: 
            count['face'] += 2 
            total['Tris'] += 2 
            total['Faces'] += 1 

      #Open Geomobject 
      file.write("*GEOMOBJECT {\n") 
      file.write("%s*NODE_NAME \"%s\"\n" % (Tab, nameMe['objName'])) 

      if nameMe.has_key('parent'): 
         file.write("%s*NODE_PARENT \"%s\"\n" % (Tab, nameMe['parent'])) 

      idnt = 1 
      mesh_matrix(file, idnt, obj, nameMe, TransTable) 

      #Open Mesh 
      file.write("%s*MESH {\n" % (Tab)) 

      idnt = 2 
      file.write("%s*TIMEVALUE 0\n" % (Tab*idnt)) 
      file.write("%s*MESH_NUMVERTEX %d\n" % ((Tab*idnt), count['vert'])) 
      file.write("%s*MESH_NUMFACES %d\n" % ((Tab*idnt), count['face'])) 

      idnt = 2 
      mesh_vertexList(file, idnt, verts, count) 
      idnt = 2 
      mesh_faceList(file, idnt, me, sGroups, faces, imgTable, hasTable, count) 


      if hasTable['hasUV'] == 1: 
         UV_list = [] 
         UVTable = {} 

         idnt = 2 
         mesh_tVertList(file, idnt, faces, UVTable, count) 
         idnt = 2 
         mesh_tFaceList(file, idnt, faces, UVTable, count) 

      if hasTable['hasVC'] == 1: 
         cVertTable = {} 

         idnt = 2 
         mesh_cVertList(file, idnt, faces, cVertTable, count) 
         idnt = 2 
         mesh_cFaceList(file, idnt, faces, cVertTable, count) 

      idnt = 2 
      mesh_normals(file, idnt, faces, verts, count) 

      # Close *MESH 
      idnt = 1 
      file.write("%s}\n" % (Tab*idnt)) 

      idnt = 1 
      mesh_footer(file, idnt, hasTable) 

      # Close *GEOMOBJECT 
      file.write("}\n") 


def mesh_matrix(file, idnt, obj, nameMe, TransTable): 

   location = obj.getLocation() 
   row = obj.getMatrix('localspace').invert() 
   quat = row.invert().toQuat() 
   rota = quat.axis 
   angle = quat.angle 

   Blender.Window.DrawProgressBar(0.0, "Writing Node Transform") 

   file.write("%s*NODE_TM {\n" % (Tab*idnt)) 

   idnt += 1 
   file.write("%s*NODE_NAME \"%s\"\n" % ((Tab*idnt), nameMe['meName'])) 
   # Inherit from what?.. 
   file.write("%s*INHERIT_POS 0 0 0\n" % (Tab*idnt)) 
   file.write("%s*INHERIT_ROT 0 0 0\n" % (Tab*idnt)) 
   file.write("%s*INHERIT_SCL 0 0 0\n" % (Tab*idnt)) 

   file.write("%s*TM_ROW0 %.4f %.4f %.4f\n" % ((Tab*idnt), row[0][0], row[0][1], row[0][2])) 
   file.write("%s*TM_ROW1 %.4f %.4f %.4f\n" % ((Tab*idnt), row[1][0], row[1][1], row[1][2])) 
   file.write("%s*TM_ROW2 %.4f %.4f %.4f\n" % ((Tab*idnt), row[2][0], row[2][1], row[2][2])) 
   file.write("%s*TM_ROW3 %.4f %.4f %.4f\n" % ((Tab*idnt), row[3][0], row[3][1], row[3][2])) 

   file.write("%s*TM_POS %.4f %.4f %.4f\n" % ((Tab*idnt), location[0], location[1], location[2])) 

   file.write("%s*TM_ROTAXIS %.4f %.4f %.4f\n" % ((Tab*idnt), rota.x, rota.y, rota.z)) 
   file.write("%s*TM_ROTANGLE %.4f\n" % ((Tab*idnt), angle)) #Look wrong, ase have lower values 

   file.write("%s*TM_SCALE %.4f %.4f %.4f\n" % ((Tab*idnt), TransTable['SizeX'], TransTable['SizeY'], TransTable['SizeZ'])) 
   #file.write("%s*TM_SCALEAXIS 0.0000 0.0000 0.0000\n" % (Tab*idnt)) 
   # Looks more logic: 
   file.write("%s*TM_SCALEAXIS %.4f %.4f %.4f\n" % ((Tab*idnt), rota.x, rota.y, rota.z)) 
   file.write("%s*TM_SCALEAXISANG %.4f\n" % ((Tab*idnt), angle)) 

   idnt -= 1 
   file.write("%s}\n" % (Tab*idnt)) 


def mesh_vertexList(file, idnt, verts, count): 

   file.write("%s*MESH_VERTEX_LIST {\n" % (Tab*idnt)) 

   idnt += 1 

   Blender.Window.DrawProgressBar(0.0, "Writing vertices") 

   for current_vert in verts: 

      vIndex = current_vert.index 

      if (vIndex % 1000) == 0: 
                   Blender.Window.DrawProgressBar((vIndex+1.0) / count['vert'], "Writing vertices") 

      file.write("%s*MESH_VERTEX %d   %.4f   %.4f   %.4f\n" % ((Tab*idnt), vIndex, current_vert.co[0], current_vert.co[1], current_vert.co[2])) 

   idnt -= 1 
   file.write("%s}\n" % (Tab*idnt)) 


def mesh_faceList(file, idnt, me, sGroups, faces, imgTable, hasTable, count): 

   file.write("%s*MESH_FACE_LIST {\n" % (Tab*idnt)) 
   idnt += 1 
   faceNo = 0 

   Blender.Window.DrawProgressBar(0.0, "Writing faces") 

   for current_face in faces: 

      face_verts = current_face.verts 
      smooth = '   *MESH_SMOOTHING' 
      matID = '   *MESH_MTLID 0' 

      if (faceNo % 500) == 0: 
         Blender.Window.DrawProgressBar((faceNo+1.0) / count['face'], "Writing faces") 

      if hasTable['hasMat'] == 1: # Blender mats 
         matID = '   *MESH_MTLID %s' % (current_face.mat) 
      elif hasTable['hasMat'] == 2: # UV fake mats 
         cur_img = current_face.image.getName() 
         matID = '   *MESH_MTLID %s' % (imgTable[cur_img]) 

      if len(face_verts) is 3: 
         vert0 = face_verts[0].index 
         vert1 = face_verts[1].index 
         vert2 = face_verts[2].index 

         #Find Smooth for this face: 
         if hasTable['hasSG'] == 1: 
            firstSmooth = 1 
            for current_SG in sGroups: 
               sGroup = me.getVertsFromGroup(current_SG) 
               if vert0 in sGroup and vert1 in sGroup and vert2 in sGroup: 
                  if firstSmooth == 1: 
                     smooth += ' %s' % (sGroups[current_SG]) 
                     firstSmooth = 0 
                  else: 
                     smooth += ', %s' % (sGroups[current_SG]) 
         elif current_face.smooth == 1: 
            smooth += ' %s' % (current_face.smooth) 

         file.write("%s*MESH_FACE %4d:    A: %4d B: %4d C: %4d AB:    1 BC:    1 CA:    1 %s %s\n" % ((Tab*idnt), faceNo, vert0, vert1, vert2, smooth, matID)) 
         faceNo+=1 

      elif len(face_verts) is 4: 
         vert0 = face_verts[0].index 
         vert1 = face_verts[1].index 
         vert2 = face_verts[2].index 
         vert3 = face_verts[3].index 

         #Find Smooth for this face: 
         if guiTable['VG2SG'] == 1 and hasTable['hasSG'] == 1: 
            firstSmooth = 1 
            for current_SG in sGroups: 
               sGroup = me.getVertsFromGroup(current_SG) 
               if vert0 in sGroup and vert1 in sGroup and vert2 in sGroup and vert3 in sGroup: 
                  if firstSmooth == 1: 
                     smooth += ' %s' % (sGroups[current_SG]) 
                     firstSmooth = 0 
                  else: 
                     smooth += ', %s' % (sGroups[current_SG]) 

         elif current_face.smooth: 
            smooth += ' %s' % (current_face.smooth) 

         file.write("%s*MESH_FACE %4d:    A: %4d B: %4d C: %4d AB:    1 BC:    1 CA:    0 %s %s\n" % ((Tab*idnt), faceNo, vert0, vert1, vert2, smooth, matID)) 
         faceNo+=1 
         file.write("%s*MESH_FACE %4d:    A: %4d B: %4d C: %4d AB:    1 BC:    1 CA:    0 %s %s\n" % ((Tab*idnt), faceNo, vert2, vert3, vert0, smooth, matID)) 
         faceNo+=1 

   idnt -= 1 
   file.write("%s}\n" % (Tab*idnt)) 


def mesh_tVertList(file, idnt, faces, UVTable, count): 

   UV_idx = 0 
   cfacetmp = 1 

   Blender.Window.DrawProgressBar(0.0, "Setup UV vertices") 

   for current_face in faces: 

      for current_uv in current_face.uv: 

         if not UVTable.has_key((current_uv.x, current_uv.y)): 
            UVTable[current_uv.x, current_uv.y] = UV_idx 
            UV_idx += 1 

      cfacetmp += 1 

   count['UVs'] = len(UVTable) 
   file.write("%s*MESH_NUMTVERTEX %d\n" % ((Tab*idnt), count['UVs'])) 
   file.write("%s*MESH_TVERTLIST {\n" % (Tab*idnt)) 

   idnt += 1 
   Blender.Window.DrawProgressBar(0.0, "Writing UV vertices") 

   UV_idx = 0 
   while UV_idx < count['UVs']: 
      if (UV_idx % 1000) == 0: 
         Blender.Window.DrawProgressBar((UV_idx+1.0) / count['face'], "Writing UV vertices") 

      for current_UV, value in UVTable.iteritems(): 
         if value == UV_idx: 
            file.write("%s*MESH_TVERT %d    %.4f    %.4f    0.0000\n" % ((Tab*idnt), UVTable[current_UV], current_UV[0], current_UV[1])) 
            UV_idx += 1 
   idnt -= 1 
   file.write("%s}\n" % (Tab*idnt)) 


def mesh_tFaceList(file, idnt, faces, UVTable, count): 

   tfaceNo = 0 

   Blender.Window.DrawProgressBar(0.0, "Writing UV binds") 

   file.write("%s*MESH_NUMTVFACES %d\n" % ((Tab*idnt), count['face'])) 
   file.write("%s*MESH_TFACELIST {\n" % (Tab*idnt)) 

   idnt += 1 

   for current_face in faces: 

      faceUV = current_face.uv 

      if (tfaceNo % 1000) == 0: 
         Blender.Window.DrawProgressBar((tfaceNo+1.0) / count['face'], "Writing UV binds") 

      if len(faceUV) is 3: #tri 
         UV0 = UVTable[(faceUV[0].x, faceUV[0].y)] 
         UV1 = UVTable[(faceUV[1].x, faceUV[1].y)] 
         UV2 = UVTable[(faceUV[2].x, faceUV[2].y)] 
         file.write("%s*MESH_TFACE %d   %d   %d   %d\n" % ((Tab*idnt), tfaceNo, UV0, UV1, UV2)) 
         tfaceNo+=1 

      elif len(faceUV) is 4: #quad 
         UV0 = UVTable[(faceUV[0].x, faceUV[0].y)] 
         UV1 = UVTable[(faceUV[1].x, faceUV[1].y)] 
         UV2 = UVTable[(faceUV[2].x, faceUV[2].y)] 
         UV3 = UVTable[(faceUV[3].x, faceUV[3].y)] 
         file.write("%s*MESH_TFACE %d   %d   %d   %d\n" % ((Tab*idnt), tfaceNo, UV0, UV1, UV2)) 
         tfaceNo+=1 
         file.write("%s*MESH_TFACE %d   %d   %d   %d\n" % ((Tab*idnt), tfaceNo, UV2, UV3, UV0)) 
         tfaceNo+=1 

   idnt -= 1 
   file.write("%s}\n" % (Tab*idnt)) 


def mesh_cVertList(file, idnt, faces, cVertTable, count): 

   Blender.Window.DrawProgressBar(0.0, "Setup VCol index") 

   for current_face in faces: 

      for current_col in current_face.col: 

         if not cVertTable.has_key((current_col.r, current_col.g, current_col.b)): 
            cVertTable[current_col.r, current_col.g, current_col.b] = count['cVert'] 
            count['cVert'] += 1 

   file.write("%s*MESH_NUMCVERTEX %d\n" % ((Tab*idnt), count['cVert'])) 
   file.write("%s*MESH_CVERTLIST {\n" % (Tab*idnt)) 

   idnt += 1 

   Blender.Window.DrawProgressBar(0.0, "Writing VCol index") 

   cV_idx = 0 

   while cV_idx < count['cVert']: 
      if (cV_idx % 1000) == 0: 
         Blender.Window.DrawProgressBar((cV_idx+1.0) / count['cVert'], "Writing UV vertices") 

      for current_cvert, value in cVertTable.iteritems(): 
         if value == cV_idx: 

            file.write("%s*MESH_VERTCOL %d   %.4f   %.4f   %.4f\n" % ((Tab*idnt), cVertTable[current_cvert], (current_cvert[0]/256.), (current_cvert[1]/256.), (current_cvert[2]/256.))) 
            cV_idx += 1 

   idnt -= 1 
   file.write("%s}\n" % (Tab*idnt)) 


def mesh_cFaceList(file, idnt, faces, cVertTable, count): 

   cFaceNo = 0 

   Blender.Window.DrawProgressBar(0.0, "Writing VCol binds") 

   file.write("%s*MESH_NUMCFACES %d\n" % ((Tab*idnt), count['face'])) 
   file.write("%s*MESH_CFACELIST {\n" % (Tab*idnt)) 

   idnt += 1 
   for current_face in faces: 

      if (cFaceNo % 500) == 0: 
         Blender.Window.DrawProgressBar((cFaceNo+1.0) / count['face'], "Writing VCol binds") 

      if len(current_face.verts) is 3: #tri 
         color0 = cVertTable[(current_face.col[0].r, current_face.col[0].g, current_face.col[0].b)] 
         color1 = cVertTable[(current_face.col[1].r, current_face.col[1].g, current_face.col[1].b)] 
         color2 = cVertTable[(current_face.col[2].r, current_face.col[2].g, current_face.col[2].b)] 

         file.write("%s*MESH_CFACE %d   %d   %d   %d\n" % ((Tab*idnt), cFaceNo, color0, color1, color2)) 
         cFaceNo+= 1 

      elif len(current_face.verts) is 4: #quad 
         color0 = cVertTable[(current_face.col[0].r, current_face.col[0].g, current_face.col[0].b)] 
         color1 = cVertTable[(current_face.col[1].r, current_face.col[1].g, current_face.col[1].b)] 
         color2 = cVertTable[(current_face.col[2].r, current_face.col[2].g, current_face.col[2].b)] 
         color3 = cVertTable[(current_face.col[3].r, current_face.col[3].g, current_face.col[3].b)] 

         file.write("%s*MESH_CFACE %d   %d   %d   %d\n" % ((Tab*idnt), cFaceNo, color0, color1, color2)) 
         cFaceNo+= 1 
         file.write("%s*MESH_CFACE %d   %d   %d   %d\n" % ((Tab*idnt), cFaceNo, color2, color3, color0)) 
         cFaceNo+= 1 

   idnt -= 1 
   file.write("%s}\n" % (Tab*idnt)) 


def mesh_normals(file, idnt, faces, verts, count): 

   # To export quads it is needed to calculate all face and vertex normals new! 

   vNorms = [] 
   fNorms = [] 

   file.write("%s*MESH_NORMALS {\n" % (Tab*idnt)) 

   Blender.Window.DrawProgressBar(0.0, "Setup Normals") 

   #-- Make empty vectors for all vertices 
   for current_vert in verts: 
      void_Vec = Blender.Mathutils.Vector(0.0, 0.0, 0.0) 
      vNorms.append(void_Vec) 

   #-- Calculate new face and vertex normals 

   for current_face in faces: 

      face_verts = current_face.verts 

      if len(face_verts) is 3: #tri 
         vert0 = face_verts[0] 
         vert1 = face_verts[1] 
         vert2 = face_verts[2] 

         face_norm = Blender.Mathutils.TriangleNormal(vert0.co, vert1.co, vert2.co) 

         fNorms.append(face_norm) 

         vNorms[vert0.index] = vNorms[vert0.index] + face_norm 
         vNorms[vert1.index] = vNorms[vert1.index] + face_norm 
         vNorms[vert2.index] = vNorms[vert2.index] + face_norm 

      elif len(face_verts) is 4: #quad 
         vert0 = face_verts[0] 
         vert1 = face_verts[1] 
         vert2 = face_verts[2] 
         vert3 = face_verts[3] 

         #---- Face Normal 
         face_norm0 = Blender.Mathutils.TriangleNormal(vert0.co, vert1.co, vert2.co) 
         face_norm1 = Blender.Mathutils.TriangleNormal(vert2.co, vert3.co, vert0.co) 

         fNorms.append(face_norm0) 
         fNorms.append(face_norm1) 

         #---- Vertex Normal 
         vNorms[vert0.index] = vNorms[vert0.index] + face_norm0 
         vNorms[vert1.index] = vNorms[vert1.index] + face_norm0 
         vNorms[vert2.index] = vNorms[vert2.index] + face_norm0 

         vNorms[vert2.index] = vNorms[vert2.index] + face_norm1 
         vNorms[vert3.index] = vNorms[vert3.index] + face_norm1 
         vNorms[vert0.index] = vNorms[vert0.index] + face_norm1 

   #-- Normalize vectors 
   for current_vec in vNorms: 
      current_vec.normalize() 

   #-- Finally write normals 
   normNo = 0 
   idnt += 2 

   Blender.Window.DrawProgressBar(0.0, "Writing Normals") 

   for current_face in faces: 

      if (normNo % 500) == 0: 
         Blender.Window.DrawProgressBar((normNo+1.0) / count['face'], "Writing Normals") 

      face_verts = current_face.verts 

      if len(face_verts) is 3: #tri 
         vert0 = face_verts[0].index 
         vert1 = face_verts[1].index 
         vert2 = face_verts[2].index 

         idnt -= 1 
         face_norm = fNorms[normNo] 
         file.write("%s*MESH_FACENORMAL %d   %.4f   %.4f   %.4f\n" % ((Tab*idnt), normNo, face_norm.x, face_norm.y, face_norm.z)) 
         normNo += 1 

         idnt += 1 
         mesh_vertNorm(file, idnt, vert0, vert1, vert2, vNorms) 

      elif len(face_verts) is 4: #quad 
         vert0 = face_verts[0].index 
         vert1 = face_verts[1].index 
         vert2 = face_verts[2].index 
         vert3 = face_verts[3].index 

         idnt -= 1 
         face_norm0 = fNorms[normNo] 
         file.write("%s*MESH_FACENORMAL %d   %.4f   %.4f   %.4f\n" % ((Tab*idnt), normNo, face_norm0.x, face_norm0.y, face_norm0.z)) 
         normNo += 1 

         idnt += 1 
         mesh_vertNorm(file, idnt, vert0, vert1, vert2, vNorms) 

         idnt -= 1 
         face_norm1 = fNorms[normNo] 
         file.write("%s*MESH_FACENORMAL %d   %.4f   %.4f   %.4f\n" % ((Tab*idnt), normNo, face_norm1.x, face_norm1.y, face_norm1.z)) 
         normNo += 1 

         idnt += 1 
         mesh_vertNorm(file, idnt, vert2, vert3, vert0, vNorms) 

   idnt -= 2 
   file.write("%s}\n" % (Tab*idnt)) 
    
def mesh_vertNorm(file, idnt, v0, v1, v2, vNorms): 

   vert_norm0 = vNorms[v0] 
   vert_norm1 = vNorms[v1] 
   vert_norm2 = vNorms[v2] 

   file.write("%s*MESH_VERTEXNORMAL %d   %.4f   %.4f   %.4f\n" % ((Tab*idnt), v0, vert_norm0.x, vert_norm0.y, vert_norm0.z)) 
   file.write("%s*MESH_VERTEXNORMAL %d   %.4f   %.4f   %.4f\n" % ((Tab*idnt), v1, vert_norm1.x, vert_norm1.y, vert_norm1.z)) 
   file.write("%s*MESH_VERTEXNORMAL %d   %.4f   %.4f   %.4f\n" % ((Tab*idnt), v2, vert_norm2.x, vert_norm2.y, vert_norm2.z)) 


def mesh_footer(file, idnt, hasTable): 

   file.write("%s*PROP_MOTIONBLUR 0\n" % (Tab*idnt)) 
   file.write("%s*PROP_CASTSHADOW 1\n" % (Tab*idnt)) 
   file.write("%s*PROP_RECVSHADOW 1\n" % (Tab*idnt)) 

   if hasTable['hasMat'] != 0: 
      file.write("%s*MATERIAL_REF %d\n" % ((Tab*idnt), hasTable['matRef'])) 

   #-------------------------End?---------------------- 


def write_ui(filename): 

   global guiTable, EXPORT_MOD, EXPORT_MTL, EXPORT_UV, EXPORT_VC, EXPORT_SELO, EXPORT_UVI, EXPORT_VG2SG 
   guiTable = {'MOD': 0, 'MTL': 1, 'UV': 1, 'VC': 1, 'SELO': 1, 'UVI': 0, 'VG2SG': 1} 

   EXPORT_MOD = Draw.Create(guiTable['MOD']) 
   EXPORT_MTL = Draw.Create(guiTable['MTL']) 
   EXPORT_UV = Draw.Create(guiTable['UV']) 
   EXPORT_VC = Draw.Create(guiTable['VC']) 
   EXPORT_SELO = Draw.Create(guiTable['SELO']) 
   EXPORT_UVI = Draw.Create(guiTable['UVI']) 
   EXPORT_VG2SG = Draw.Create(guiTable['VG2SG']) 

   # Get USER Options 
   pup_block = [('Mesh Options...'),('Apply Modifiers', EXPORT_MOD, 'Use modified mesh data from each object.'),('Materials', EXPORT_MTL, 'Export Materials.'),('Face UV', EXPORT_UV, 'Export texface UV coords.'),('Vertex Colors', EXPORT_VC, 'Export vertex colors'),('Context...'),('Selection Only', EXPORT_SELO, 'Only export objects in visible selection, else export all mesh object.'),('Bonus...'),('UV Image as Material', EXPORT_UVI, 'Convert UV Images to Materials.'),('VertGr. as SmoothGr.', EXPORT_VG2SG, 'Make SmoothGroups by VertGroups. See doc.')] 

   if not Draw.PupBlock('Export...', pup_block): 
      return 

   Window.WaitCursor(1) 

   guiTable['MOD'] = EXPORT_MOD.val 
   guiTable['MTL'] = EXPORT_MTL.val 
   guiTable['UV'] = EXPORT_UV.val 
   guiTable['VC'] = EXPORT_VC.val 
   guiTable['SELO'] = EXPORT_SELO.val 
   guiTable['UVI'] = EXPORT_UVI.val 
   guiTable['VG2SG'] = EXPORT_VG2SG.val 

   #Recheck 
   if filename.find('.ase', -4) <= 0: 
      filename += '.ase' 

   write(filename) 

   Window.WaitCursor(0) 


if __name__ == '__main__': 
   Window.FileSelector(write_ui, 'Export ASCII Scene', sys.makename(ext='.ase'))